package pt.caf_pbs.schooldiskapi.controller;


import org.springframework.beans.BeanUtils;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.server.ResponseStatusException;
import pt.caf_pbs.schooldiskapi.dao.FilesOperation;
import pt.caf_pbs.schooldiskapi.dto.InstructorDTO;
import pt.caf_pbs.schooldiskapi.dto.InstructorEditDTO;
import pt.caf_pbs.schooldiskapi.dto.LectureDTO;
import pt.caf_pbs.schooldiskapi.dto.SubjectDTO;
import pt.caf_pbs.schooldiskapi.exceptions.DuplicatedDataException;
import pt.caf_pbs.schooldiskapi.model.Instructor;
import pt.caf_pbs.schooldiskapi.model.Lecture;
import pt.caf_pbs.schooldiskapi.model.School;
import pt.caf_pbs.schooldiskapi.model.Subject;

import java.io.File;
import java.util.List;


@RestController
@RequestMapping("/v1/instructor")
public class InstructorsController {

    @PutMapping("{initials}")
    public void updateInstructor(@PathVariable String initials, @RequestBody InstructorDTO instructorDTO) {
        School school = FilesOperation.loadSchool();
        school.getInstructorContainer().update(instructorDTO.getInitials(), instructorDTO.getName());
        FilesOperation.storeSchool(school);
    }

    @PostMapping
    public Instructor createInstructor(@RequestBody InstructorEditDTO instructorDTO) {
        try {
            School school = FilesOperation.loadSchool();
            Instructor instructor = new Instructor(instructorDTO.getInitials(), instructorDTO.getName());
            school.getInstructorContainer().add(instructor);
            FilesOperation.storeSchool(school);
            return instructor;
        }catch(DuplicatedDataException e){
            throw new ResponseStatusException(HttpStatus.CONFLICT, "Problems with "+e.getMessage(), e);
        }
    }

    @GetMapping
    public List<Instructor> getInstructors() {
        School school = FilesOperation.loadSchool();
        return school.getInstructorContainer().getInstructors();
    }

    @GetMapping("{initials}")
    public Instructor getInstructor(@PathVariable String initials) {
        School school = FilesOperation.loadSchool();
        Instructor instructor = school.getInstructorContainer().get(initials);
        if(instructor != null){
            return instructor;
        }else{
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Instructor with initials " + initials + " found");
        }
    }

    @DeleteMapping("{initials}")
    public void deleteInstructor(@PathVariable String initials) {
        School school = FilesOperation.loadSchool();
        Instructor instructor = school.getInstructorContainer().get(initials);
        if(instructor != null){
            school.getInstructorContainer().remove(initials);
            FilesOperation.storeSchool(school);
        }else{
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Instructor with initials " + initials + " found");
        }
    }

    @GetMapping( path = "{instructorInitials}/lecture/{subjectInitials}")
    public Lecture getLecture(@PathVariable String instructorInitials, @PathVariable String subjectInitials) {
        School school = FilesOperation.loadSchool();
        Instructor instructor = school.getInstructorContainer().get(instructorInitials);
        if(instructor != null){
            Lecture lecture = instructor.getLectures().get(subjectInitials);
            if(lecture != null) {
                return lecture;
            }else{
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No subject with lecture with initials " + subjectInitials + " found");
            }
        }else{
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Instructor with initials " + instructorInitials + " found");
        }
    }

    @PostMapping( path = "{instructorInitials}/lecture")
    public Lecture createLecture(@PathVariable String instructorInitials, @RequestBody LectureDTO lectureDTO) {
        School school = FilesOperation.loadSchool();
        try {
            Instructor instructor = school.getInstructorContainer().get(instructorInitials);
            if (instructor != null) {
                Lecture lecture = new Lecture();
                Subject subject = new Subject(lectureDTO.getSubject());
                lecture.setSubject(subject);
                instructor.getLectures().add(lecture);
                FilesOperation.storeSchool(school);
                return lecture;
            } else {
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Instructor with initials " + instructorInitials + " found");
            }
        }catch(DuplicatedDataException e){
            throw new ResponseStatusException(HttpStatus.CONFLICT, "Problems with "+e.getMessage(), e);
        }
    }

    @PutMapping( path = "{instructorInitials}/lecture/{subjectInitials}")
    public Lecture updateLecture(@PathVariable String instructorInitials, @PathVariable String subjectInitials, @RequestBody LectureDTO lectureDTO) {
        School school = FilesOperation.loadSchool();
        try {
            Instructor instructor = school.getInstructorContainer().get(instructorInitials);
            if (instructor != null) {
                Lecture lecture = instructor.getLectures().get(subjectInitials);
                if(lecture != null) {
                    Subject subject = new Subject(lectureDTO.getSubject());
                    lecture.setSubject(subject);
                    FilesOperation.storeSchool(school);
                    return lecture;
                }else{
                    throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No lecture with subject initials " + subjectInitials + " found");
                }
            } else {
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Instructor with initials " + instructorInitials + " found");
            }
        }catch(DuplicatedDataException e){
            throw new ResponseStatusException(HttpStatus.CONFLICT, "Problems with "+e.getMessage(), e);
        }
    }

    @DeleteMapping( path = "{instructorInitials}/lecture/{subjectInitials}")
    public Lecture deleteLecture(@PathVariable String instructorInitials, @PathVariable String subjectInitials) {
        School school = FilesOperation.loadSchool();
        Instructor instructor = school.getInstructorContainer().get(instructorInitials);
        if (instructor != null) {
            Lecture lecture = instructor.getLectures().get(subjectInitials);
            if (lecture != null) {
                instructor.getLectures().remove(subjectInitials);
                FilesOperation.storeSchool(school);
                return lecture;
            } else {
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No lecture with subject initials " + subjectInitials + " found");
            }
        } else {
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Instructor with initials " + instructorInitials + " found");
        }
    }
}
